import xarray as xr
import numpy as np
import matplotlib.pyplot as plt

# -----------------------------
# USER FILE PATHS
# -----------------------------
BEDMACHINE_FILE = "bedmachine4.nc"
VELOCITY_FILE   = "antarctica_velocity_measures.nc"

# -----------------------------
# Load datasets (force CF decoding)
# -----------------------------
bm = xr.open_dataset(BEDMACHINE_FILE, decode_cf=True)
vel = xr.open_dataset(VELOCITY_FILE, decode_cf=True)

surface = bm["surface"]
thickness = bm["thickness"]
mask = bm["mask"]

# -----------------------------
# Identify velocity components robustly
# -----------------------------
vx_name = [v for v in vel.data_vars if v.lower() in ("vx", "vx_surface", "vx_measured") or "vx" in v.lower()][0]
vy_name = [v for v in vel.data_vars if v.lower() in ("vy", "vy_surface", "vy_measured") or "vy" in v.lower()][0]

vx = vel[vx_name].astype("float64")
vy = vel[vy_name].astype("float64")

# -----------------------------
# Compute speed safely
# -----------------------------
speed = np.sqrt(vx * vx + vy * vy)

# -----------------------------
# Regrid velocity to BedMachine grid if needed
# -----------------------------
if speed.shape != thickness.shape:
    speed = speed.interp_like(thickness, method="nearest")

# -----------------------------
# Grid cell area
# -----------------------------
dx = float(np.abs(bm["x"][1] - bm["x"][0]))
dy = float(np.abs(bm["y"][1] - bm["y"][0]))
cell_area = dx * dy  # m²

# -----------------------------
# Velocity-weighted effective volume
# -----------------------------
vmin = 1.0  # m/yr floor to avoid singularities
effective_volume = (thickness * cell_area) / xr.where(speed > vmin, speed, vmin)

effective_volume = effective_volume.where(thickness > 0)

# -----------------------------
# Age-proxy bins
# -----------------------------
def bin_sum(cond):
    return effective_volume.where(cond, 0.0).sum().item()

bins = {
    ">3000 m (~100 kyr – 1+ Myr)": (mask == 2) & (surface > 3000),
    "2000–3000 m (~10–100 kyr)":  (mask == 2) & (surface > 2000) & (surface <= 3000),
    "1000–2000 m (~1–10 kyr)":    (mask == 2) & (surface > 1000) & (surface <= 2000),
    "<1000 m grounded (~<1 kyr)": (mask == 2) & (surface <= 1000),
    "Floating ice (~<1 kyr)":     (mask == 3),
}

bin_values = {k: bin_sum(v) for k, v in bins.items()}
total_eff = sum(bin_values.values())

fractions = {k: 100 * v / total_eff for k, v in bin_values.items()}

# -----------------------------
# Print results
# -----------------------------
print("\nVelocity-weighted effective volume fractions:\n")
for k, v in fractions.items():
    print(f"{k:35s}: {v:6.2f} %")

# -----------------------------
# Plot (black & white)
# -----------------------------
labels = list(fractions.keys())
values = np.array(list(fractions.values()))
y = np.arange(len(labels))

fig, ax = plt.subplots(figsize=(7, 4.5))

ax.errorbar(
    values,
    y,
    xerr=None,
    fmt="o",
    color="black",
    ecolor="black",
    elinewidth=2,
    capsize=0
)

ax.hlines(y, 0, values, color="black", linewidth=2)

ax.set_yticks(y)
ax.set_yticklabels(labels)
ax.invert_yaxis()
ax.set_xlabel("Velocity-weighted share of Antarctic ice volume (%)")
ax.set_title("Antarctic Ice Volume Weighted by Surface Flow Speed")

ax.grid(axis="x", linestyle=":", linewidth=0.8)

plt.savefig(
    "antarctica_velocity_weighted_volume.png",
    dpi=300,
    bbox_inches="tight"
)
plt.close()

print("\nSaved figure: antarctica_velocity_weighted_volume.png")
