import pandas as pd
import matplotlib.pyplot as plt
import numpy as np

# -----------------------------
# INPUT
# -----------------------------
CSV_FILE = "antarctica_age_exceedance_volume_table_corrected.csv"

df = pd.read_csv(CSV_FILE).sort_values("Threshold_yr")

# -----------------------------
# Threshold definitions
# -----------------------------
central_T = 10000
lower_T   = 5000
upper_T   = 20000

regions = ["Antarctica", "EAIS", "WAIS"]

# -----------------------------
# Extract bounds safely
# -----------------------------
results = {}

for r in regions:
    v5  = df.loc[df["Threshold_yr"] == lower_T, r].values[0]
    v10 = df.loc[df["Threshold_yr"] == central_T, r].values[0]
    v20 = df.loc[df["Threshold_yr"] == upper_T, r].values[0]

    lo = min(v5, v20)
    hi = max(v5, v20)

    results[r] = dict(
        central=v10,
        lo=lo,
        hi=hi
    )

# -----------------------------
# Print summary
# -----------------------------
print("\nCentral estimates and uncertainty bounds (5–20 kyr):\n")

for r, d in results.items():
    print(f"{r:12s}: {d['central']:.1f}%   [{d['lo']:.1f} – {d['hi']:.1f}] %")

# -----------------------------
# Prepare plot data
# -----------------------------
x = np.arange(len(regions))
central_vals = [results[r]["central"] for r in regions]
lo_vals = [results[r]["lo"] for r in regions]
hi_vals = [results[r]["hi"] for r in regions]

# -----------------------------
# Plot
# -----------------------------
fig, ax = plt.subplots(figsize=(6, 4))

for i in range(len(regions)):
    # vertical envelope
    ax.vlines(
        x[i],
        lo_vals[i],
        hi_vals[i],
        color="black",
        linewidth=3
    )
    # central estimate
    ax.plot(
        x[i],
        central_vals[i],
        "o",
        color="black"
    )

ax.set_xticks(x)
ax.set_xticklabels(regions)
ax.set_ylabel("Ice volume fraction exceeding 10 kyr (%)")

ax.set_title(
    "Error-bounded volumetric age exceedance\n"
    "Central estimate at 10 kyr; envelope from 5–20 kyr"
)

ax.grid(axis="y", linestyle=":", linewidth=0.8)

plt.tight_layout()
plt.savefig(
    "antarctica_age_exceedance_envelope.png",
    dpi=300,
    bbox_inches="tight"
)
plt.close()

print("\nSaved: antarctica_age_exceedance_envelope.png")
