#!/usr/bin/env python3
"""
Plotting script for site-based diagnostics and model event-rate envelope.

Inputs:
- site_sampling_timeseries.csv
- model_event_rate_envelope.csv

Outputs:
- fig_sites_dzdt_timeseries.png
- fig_sites_zero_contour_distance.png
- fig_sites_zero_contour_variance.png
- fig_model_event_rate_envelope.png
"""

import pandas as pd
import matplotlib.pyplot as plt

# ============================================================
# LOAD DATA
# ============================================================

df_sites = pd.read_csv("site_sampling_timeseries.csv")
df_event = pd.read_csv("model_event_rate_envelope.csv")

# Split groups
df_homo = df_sites[df_sites["group"] == "Homo"]
df_civ  = df_sites[df_sites["group"] == "Civilization"]

# ============================================================
# FIGURE 1 — MEAN |ΔZeff/Δt| AT SITES
# ============================================================

plt.figure(figsize=(7,4))

plt.plot(
    df_homo["time_ka"],
    df_homo["mean_abs_dzdt"],
    label="Early Homo",
    linewidth=2
)

plt.plot(
    df_civ["time_ka"],
    df_civ["mean_abs_dzdt"],
    label="Early Civilizations",
    linewidth=2
)

plt.gca().invert_xaxis()
plt.xlabel("ka BP")
plt.ylabel("Mean |ΔZeff/Δt| (m / yr)")
plt.title("Surface adjustment rate at sites")
plt.grid(alpha=0.3)
plt.legend(frameon=False)

plt.tight_layout()
plt.savefig("fig_sites_dzdt_timeseries.png", dpi=300)
plt.close()

# ============================================================
# FIGURE 2 — MEAN ZERO-CONTOUR DISTANCE
# ============================================================

plt.figure(figsize=(7,4))

plt.plot(
    df_homo["time_ka"],
    df_homo["mean_zero_contour_distance_deg"],
    label="Early Homo",
    linewidth=2
)

plt.plot(
    df_civ["time_ka"],
    df_civ["mean_zero_contour_distance_deg"],
    label="Early Civilizations",
    linewidth=2
)

plt.gca().invert_xaxis()
plt.xlabel("ka BP")
plt.ylabel("Mean distance to equilibrium margin (deg)")
plt.title("Proximity of sites to equilibrium margin")
plt.grid(alpha=0.3)
plt.legend(frameon=False)

plt.tight_layout()
plt.savefig("fig_sites_zero_contour_distance.png", dpi=300)
plt.close()

# ============================================================
# FIGURE 3 — VARIANCE OF ZERO-CONTOUR DISTANCE
# ============================================================

plt.figure(figsize=(7,4))

plt.plot(
    df_homo["time_ka"],
    df_homo["var_zero_contour_distance_deg2"],
    label="Early Homo",
    linewidth=2
)

plt.plot(
    df_civ["time_ka"],
    df_civ["var_zero_contour_distance_deg2"],
    label="Early Civilizations",
    linewidth=2
)

plt.gca().invert_xaxis()
plt.xlabel("ka BP")
plt.ylabel("Variance of distance (deg²)")
plt.title("Spatial stability of site environments")
plt.grid(alpha=0.3)
plt.legend(frameon=False)

plt.tight_layout()
plt.savefig("fig_sites_zero_contour_variance.png", dpi=300)
plt.close()

# ============================================================
# FIGURE 4 — MODEL-DERIVED EVENT-RATE ENVELOPE
# ============================================================

plt.figure(figsize=(7,4))

plt.plot(
    df_event["time_ka"],
    df_event["event_rate_proxy"],
    linewidth=2
)

plt.gca().invert_xaxis()
plt.xlabel("ka BP")
plt.ylabel("Normalized event-rate proxy")
plt.title("Model-derived global event-rate envelope")
plt.grid(alpha=0.3)

plt.tight_layout()
plt.savefig("fig_model_event_rate_envelope.png", dpi=300)
plt.close()

print("✓ Site and event-rate diagnostic figures generated")
