#!/usr/bin/env python3
"""
Plotting script for temporal phase randomization null test.

Inputs:
- temporal_null_distributions.csv
- temporal_null_summary.csv

Outputs:
- fig_temporal_null_homo_dzdt.png
- fig_temporal_null_homo_variance.png
- fig_temporal_null_civ_dzdt.png
- fig_temporal_null_civ_variance.png
"""

import pandas as pd
import matplotlib.pyplot as plt

# ============================================================
# LOAD DATA
# ============================================================

df_null = pd.read_csv("temporal_null_distributions.csv")
df_sum  = pd.read_csv("temporal_null_summary.csv")

def get_obs(metric):
    return df_sum[df_sum["metric"] == metric]["observed_alignment"].values[0]

def plot_null(null_vals, obs_val, title, fname):
    plt.figure(figsize=(6,4))
    plt.hist(null_vals, bins=40, density=True, alpha=0.7)
    plt.axvline(obs_val, linestyle="--", linewidth=2)
    plt.xlabel("Phase alignment metric")
    plt.ylabel("Probability density")
    plt.title(title)
    plt.grid(alpha=0.3)
    plt.tight_layout()
    plt.savefig(fname, dpi=300)
    plt.close()

# ============================================================
# PLOTS
# ============================================================

plot_null(
    df_null["homo_dzdt"],
    get_obs("homo_dzdt"),
    "Temporal null: Early Homo |ΔZeff/Δt|",
    "fig_temporal_null_homo_dzdt.png"
)

plot_null(
    df_null["homo_var"],
    get_obs("homo_var"),
    "Temporal null: Early Homo stability variance",
    "fig_temporal_null_homo_variance.png"
)

plot_null(
    df_null["civ_dzdt"],
    get_obs("civ_dzdt"),
    "Temporal null: Early Civilizations |ΔZeff/ΔZt|",
    "fig_temporal_null_civ_dzdt.png"
)

plot_null(
    df_null["civ_var"],
    get_obs("civ_var"),
    "Temporal null: Early Civilizations stability variance",
    "fig_temporal_null_civ_variance.png"
)

print("✓ Temporal null-test figures generated")
