#!/usr/bin/env python3

import numpy as np
import xarray as xr

# ============================================================
# USER PARAMETERS
# ============================================================

INPUT_NC  = "gebco_tpw_effective_elevation.nc"
OUTPUT_NC = "tpw_centrifugal_relaxation_95pct_10steps.nc"

ZVAR = "effective_elevation"

# Downsample factor (15″ → 2′)
COARSEN = 8

# TPW parameters
TPW_DEG = 104.0
TPW_AZIMUTH_DEG = 31.0

# Relaxation
NSTEPS = 10
RELAX_FINAL = 0.95

# Target present-day equator-pole relief (meters)
TARGET_RELIEF = 21000.0

# ============================================================
# LOAD & DOWNSAMPLE DATA
# ============================================================

ds = xr.open_dataset(INPUT_NC)

ds = ds.coarsen(
    lat=COARSEN,
    lon=COARSEN,
    boundary="trim"
).mean()

lat = np.deg2rad(ds.lat.values)
lon = np.deg2rad(ds.lon.values)

LON, LAT = np.meshgrid(lon, lat)

# Original rotated bulge
H0 = ds[ZVAR].values

# Remove mean to conserve volume
H0 = H0 - np.nanmean(H0)

# ============================================================
# CONSTRUCT NEW SPIN AXIS (EARTH-FIXED)
# ============================================================

theta = np.deg2rad(TPW_DEG)
az    = np.deg2rad(TPW_AZIMUTH_DEG)

omega_hat = np.array([
    np.sin(theta) * np.cos(az),
    np.sin(theta) * np.sin(az),
    np.cos(theta)
])

# ============================================================
# COMPUTE NEW EQUILIBRIUM CENTRIFUGAL BULGE
# ============================================================

# Unit position vectors
x = np.cos(LAT) * np.cos(LON)
y = np.cos(LAT) * np.sin(LON)
z = np.sin(LAT)

r = np.stack([x, y, z], axis=0)

# Dot product with new spin axis
dot = (
    r[0] * omega_hat[0]
  + r[1] * omega_hat[1]
  + r[2] * omega_hat[2]
)

r_perp2 = 1.0 - dot**2

# Remove mean (volume conservation)
He = r_perp2 - np.mean(r_perp2)

# Scale to target relief
current_relief = np.nanmax(He) - np.nanmin(He)
He *= TARGET_RELIEF / current_relief

# ============================================================
# BUILD RELAXATION SEQUENCE
# ============================================================

steps = np.arange(NSTEPS)
fvals = np.linspace(0.0, RELAX_FINAL, NSTEPS)

Hseq = np.zeros((NSTEPS, *H0.shape), dtype=np.float32)

for i, f in enumerate(fvals):
    H = (1.0 - f) * H0 + f * He
    H -= np.mean(H)  # enforce zero mean each step
    Hseq[i] = H

# ============================================================
# OUTPUT NETCDF (PANOPLY-FRIENDLY)
# ============================================================

out = xr.Dataset(
    data_vars=dict(
        effective_elevation=(
            ("step", "lat", "lon"),
            Hseq,
            dict(
                units="m",
                long_name="Effective elevation relative to equilibrium sea level",
                description="Centrifugal viscoelastic relaxation following 104° TPW"
            )
        )
    ),
    coords=dict(
        step=("step", steps),
        lat=("lat", ds.lat.values),
        lon=("lon", ds.lon.values)
    ),
    attrs=dict(
        TPW_degrees=TPW_DEG,
        TPW_meridian=TPW_AZIMUTH_DEG,
        relaxation_final_fraction=RELAX_FINAL,
        downsample_factor=COARSEN,
        method="Blended centrifugal potential relaxation"
    )
)

out.to_netcdf(
    OUTPUT_NC,
    encoding={"effective_elevation": {"zlib": True, "complevel": 4}}
)

print(f"Wrote {OUTPUT_NC}")
