#!/usr/bin/env python3
"""
Plot GEBCO-integrated effective elevation under TPW equilibrium sea level
using the same map style as earlier discrete-contour TPW analyses.
"""

import numpy as np
import xarray as xr
import matplotlib.pyplot as plt
import cartopy.crs as ccrs
import cartopy.feature as cfeature

# ---------------------------------------------------------------------
# FILE PATH
# ---------------------------------------------------------------------

NC_FILE = "gebco_tpw_effective_elevation.nc"

# ---------------------------------------------------------------------
# LOAD DATA
# ---------------------------------------------------------------------

ds = xr.open_dataset(NC_FILE)
Z = ds["effective_elevation"]
lats = ds["lat"]
lons = ds["lon"]

# ---------------------------------------------------------------------
# CONTOUR LEVELS (match earlier analysis style)
# ---------------------------------------------------------------------

levels = np.array([
    -10000, -7500, -5000, -2500, -1000, -500, -250,
    0,
    250, 500, 1000, 2500, 5000, 7500, 10000
])

# ---------------------------------------------------------------------
# FIGURE SETUP (Robinson projection)
# ---------------------------------------------------------------------

fig = plt.figure(figsize=(16, 9))
ax = plt.axes(projection=ccrs.Robinson())

ax.set_global()

# ---------------------------------------------------------------------
# FILLED CONTOURS
# ---------------------------------------------------------------------

cf = ax.contourf(
    lons,
    lats,
    Z,
    levels=levels,
    cmap="RdBu_r",
    extend="both",
    transform=ccrs.PlateCarree()
)

# ---------------------------------------------------------------------
# ZERO CONTOUR (equilibrium shoreline)
# ---------------------------------------------------------------------

ax.contour(
    lons,
    lats,
    Z,
    levels=[0],
    colors="black",
    linewidths=1.2,
    transform=ccrs.PlateCarree()
)

# ---------------------------------------------------------------------
# MAP FEATURES
# ---------------------------------------------------------------------

ax.add_feature(cfeature.COASTLINE, linewidth=0.6)
ax.add_feature(cfeature.BORDERS, linewidth=0.4, alpha=0.7)

# ---------------------------------------------------------------------
# COLORBAR
# ---------------------------------------------------------------------

cbar = plt.colorbar(
    cf,
    orientation="horizontal",
    pad=0.06,
    aspect=40,
    shrink=0.9
)

cbar.set_label("Effective elevation relative to equilibrium sea level (m)")

# ---------------------------------------------------------------------
# TITLE
# ---------------------------------------------------------------------

ax.set_title(
    "Realized Land–Sea State After 104° True Polar Wander\n"
    "GEBCO Bedrock Elevation Integrated with Equilibrium Sea-Level Anomaly",
    fontsize=14,
    pad=14
)

# ---------------------------------------------------------------------
# OUTPUT
# ---------------------------------------------------------------------

plt.tight_layout()
plt.savefig(
    "../outputs/gebco_effective_elevation_map.png",
    dpi=300
)
plt.show()
