#!/usr/bin/env python3

import json
import numpy as np
import matplotlib.pyplot as plt

from scipy.interpolate import griddata
from shapely.geometry import LineString, Point
from pyproj import Geod

# ------------------------------------------------------------
# Load sea-level anomaly GeoJSON
# ------------------------------------------------------------

with open("TPW_signed_equilibrium_relative_sea_level.geojson", "r") as f:
    sl_data = json.load(f)

lons, lats, vals = [], [], []

for feat in sl_data["features"]:
    lon, lat = feat["geometry"]["coordinates"]
    val = feat["properties"]["net_relative_sea_level_m"]
    lons.append(lon)
    lats.append(lat)
    vals.append(val)

lons = np.array(lons)
lats = np.array(lats)
vals = np.array(vals)

# ------------------------------------------------------------
# Interpolate to regular grid
# ------------------------------------------------------------

lon_grid = np.linspace(-180, 180, 721)
lat_grid = np.linspace(-90, 90, 361)

LON, LAT = np.meshgrid(lon_grid, lat_grid)

Z = griddata(
    (lons, lats),
    vals,
    (LON, LAT),
    method="linear"
)

# ------------------------------------------------------------
# Extract zero-anomaly contour (matplotlib ≥3.8)
# ------------------------------------------------------------

fig, ax = plt.subplots()
cs = ax.contour(LON, LAT, Z, levels=[0.0])
plt.close(fig)

zero_contour_lines = []

# cs.allsegs is a list (per level) of lists of (N,2) arrays
for level_segs in cs.allsegs:
    for seg in level_segs:
        if len(seg) > 1:
            zero_contour_lines.append(LineString(seg))

if not zero_contour_lines:
    raise RuntimeError("Zero contour extraction failed — no contour segments found.")

# ------------------------------------------------------------
# Geodesic calculator
# ------------------------------------------------------------

geod = Geod(ellps="WGS84")

def geodesic_distance_km(point, line):
    """
    Conservative minimum geodesic distance from a point
    to a LineString by vertex sampling.
    """
    min_dist = np.inf
    px, py = point.x, point.y

    for lon, lat in line.coords:
        _, _, d = geod.inv(px, py, lon, lat)
        min_dist = min(min_dist, d)

    return min_dist / 1000.0  # meters → km

# ------------------------------------------------------------
# Load Homo sites
# ------------------------------------------------------------

with open("early_homo_sites.geojson", "r") as f:
    homo_data = json.load(f)

results = []

for feat in homo_data["features"]:
    lon, lat = feat["geometry"]["coordinates"]
    props = feat["properties"]

    site_point = Point(lon, lat)

    # Determine anomaly sign at site
    anomaly_here = griddata(
        (lons, lats),
        vals,
        [(lon, lat)],
        method="linear"
    )[0]

    side = "emergent" if anomaly_here < 0 else "submergent"

    # Distance to nearest zero-contour segment
    distances = [
        geodesic_distance_km(site_point, line)
        for line in zero_contour_lines
    ]

    min_distance = min(distances)

    results.append({
        "longitude": lon,
        "latitude": lat,
        "distance_to_zero_contour_km": min_distance,
        "side_of_contour": side,
        "taxon": props.get("taxon"),
        "min_ma": props.get("min_ma"),
        "max_ma": props.get("max_ma")
    })

# ------------------------------------------------------------
# Write output GeoJSON
# ------------------------------------------------------------

features = []

for r in results:
    features.append({
        "type": "Feature",
        "geometry": {
            "type": "Point",
            "coordinates": [r["longitude"], r["latitude"]]
        },
        "properties": r
    })

output = {
    "type": "FeatureCollection",
    "features": features
}

with open("early_homo_distance_to_zero_contour.geojson", "w") as f:
    json.dump(output, f, indent=2)

print("Wrote early_homo_distance_to_zero_contour.geojson")
