#!/usr/bin/env python3
"""
Visualize Monte Carlo null distribution of Spearman rho
with observed value overlaid.
"""

import numpy as np
import matplotlib.pyplot as plt

# --------------------------------------------------
# INPUT FILES
# --------------------------------------------------

NULL_RHOS_FILE = "null_rhos.npy"
OBSERVED_RHO_FILE = "observed_rho.txt"

# --------------------------------------------------
# LOAD DATA
# --------------------------------------------------

null_rhos = np.load(NULL_RHOS_FILE)

with open(OBSERVED_RHO_FILE, "r") as f:
    observed_rho = float(f.read().strip())

n_iter = len(null_rhos)

# Empirical p-value (one-sided, >= observed)
p_empirical = np.mean(null_rhos >= observed_rho)

# Conservative upper bound if zero
if p_empirical == 0.0:
    p_text = f"< {1/n_iter:.1e}"
else:
    p_text = f"= {p_empirical:.2e}"

# --------------------------------------------------
# PLOT
# --------------------------------------------------

plt.figure(figsize=(9, 5))

# Histogram of null distribution
counts, bins, _ = plt.hist(
    null_rhos,
    bins=60,
    density=True,
    alpha=0.75,
    edgecolor="black",
    linewidth=0.4,
    label="Null distribution (Monte Carlo)"
)

# Observed value
plt.axvline(
    observed_rho,
    color="red",
    linewidth=3,
    label=f"Observed ρ = {observed_rho:.3f}"
)

# Shade tail region
bin_centers = 0.5 * (bins[:-1] + bins[1:])
tail = bin_centers >= observed_rho

plt.fill_between(
    bin_centers[tail],
    counts[tail],
    alpha=0.6,
    label="Null ≥ observed"
)

# Labels
plt.xlabel("Spearman rank correlation (ρ)")
plt.ylabel("Probability density")

plt.title(
    "Monte Carlo Null Distribution of Homo Age–Distance Correlation\n"
    f"Empirical p-value {p_text}  (N = {n_iter})"
)

plt.legend(frameon=True)
plt.grid(alpha=0.25)

plt.tight_layout()
plt.savefig(
    "null_distribution_with_observed.png",
    dpi=300
)
plt.show()

# --------------------------------------------------
# CONSOLE SUMMARY
# --------------------------------------------------

print("Observed rho:", observed_rho)
print("Monte Carlo iterations:", n_iter)
print("Empirical p-value:", p_empirical)
if p_empirical == 0.0:
    print(f"Upper bound: p < {1/n_iter:.1e}")
