#!/usr/bin/env python3 

import json
import math
import numpy as np

# ---------------------------------------------------------------------
# Physical constants
# ---------------------------------------------------------------------

omega = 7.2921159e-5      # Earth rotation rate (rad/s)
R = 6.371e6               # Mean Earth radius (m)
g = 9.81                  # Gravitational acceleration (m/s^2)

rho_water = 1000.0
rho_mantle = 3300.0
airy_ratio = rho_water / rho_mantle

# Fraction of Airy isostatic compensation realized
comp_fraction = 0.4

# Centrifugal potential constant
C = (omega**2 * R**2) / (2 * g)

# ---------------------------------------------------------------------
# True Polar Wander definition
# ---------------------------------------------------------------------

# 104° rotation about the 31°E meridian
tpw_angle = math.radians(104.0)
tpw_meridian = math.radians(31.0)

def sph_to_cart(lat, lon):
    return np.array([
        math.cos(lat) * math.cos(lon),
        math.cos(lat) * math.sin(lon),
        math.sin(lat)
    ])

def cart_to_sph(v):
    x, y, z = v
    lat = math.asin(z / np.linalg.norm(v))
    lon = math.atan2(y, x)
    return lat, lon

def rotate_about_axis(v, axis, angle):
    axis = axis / np.linalg.norm(axis)
    return (
        v * math.cos(angle)
        + np.cross(axis, v) * math.sin(angle)
        + axis * np.dot(axis, v) * (1 - math.cos(angle))
    )

# Rotation axis: horizontal axis perpendicular to the 31°E meridian
rotation_axis = sph_to_cart(0.0, tpw_meridian + math.pi / 2)

# ---------------------------------------------------------------------
# Generate global grid and compute anomalies
# ---------------------------------------------------------------------

features = []

grid_step = 2.0  # degrees

for lat_deg in np.arange(-90, 90 + grid_step, grid_step):
    for lon_deg in np.arange(-180, 180 + grid_step, grid_step):

        lat1 = math.radians(lat_deg)
        lon1 = math.radians(lon_deg)

        # Original position
        v1 = sph_to_cart(lat1, lon1)

        # Rotated position (state 2)
        v2 = rotate_about_axis(v1, rotation_axis, tpw_angle)
        lat2, lon2 = cart_to_sph(v2)

        # Equilibrium centrifugal sea surface
        h_before = C * math.cos(lat1)**2
        h_after  = C * math.cos(lat2)**2

        # Raw equilibrium sea-level change
        delta_h = h_after - h_before

        # Isostatic adjustment
        uplift = comp_fraction * airy_ratio * delta_h

        # Net relative sea-level anomaly (SIGNED)
        net_rsl = delta_h - uplift

        features.append({
            "type": "Feature",
            "geometry": {
                "type": "Point",
                "coordinates": [lon_deg, lat_deg]
            },
            "properties": {
                "net_relative_sea_level_m": net_rsl,
                "equilibrium_change_m": delta_h,
                "isostatic_adjustment_m": uplift,
                "initial_latitude_deg": lat_deg,
                "final_latitude_deg": math.degrees(lat2)
            }
        })

# ---------------------------------------------------------------------
# Write GeoJSON
# ---------------------------------------------------------------------

geojson = {
    "type": "FeatureCollection",
    "features": features
}

output_path = "TPW_signed_equilibrium_relative_sea_level.geojson"

with open(output_path, "w") as f:
    json.dump(geojson, f)
