

GEBCO_NC = "GEBCO_2025_sub_ice.nc"
TPW_GEOJSON = "TPW_signed_equilibrium_relative_sea_level.geojson"
OUT_TIFF = "gebco_tpw_effective_elevation.tif"

#!/usr/bin/env python3
"""
Integrate GEBCO global topography/bathymetry with TPW equilibrium
relative sea-level anomaly and output realized effective elevation
as a NetCDF file.

Effective elevation = bedrock elevation - equilibrium sea-level anomaly

Positive values: emergent land
Negative values: submerged relative to equilibrium sea level
"""

import json
import numpy as np
import xarray as xr
from scipy.interpolate import griddata

# ---------------------------------------------------------------------
# FILE PATHS
# ---------------------------------------------------------------------

GEBCO_NC = "GEBCO_2025_sub_ice.nc"
TPW_GEOJSON = "TPW_signed_equilibrium_relative_sea_level.geojson"
OUT_NC = "gebco_tpw_effective_elevation.nc"

# ---------------------------------------------------------------------
# LOAD GEBCO NETCDF (explicit backend)
# ---------------------------------------------------------------------

ds_gebco = xr.open_dataset(GEBCO_NC, engine="netcdf4")

# GEBCO variables (per documentation)
# elevation: meters, positive = land, negative = bathymetry
elev = ds_gebco["elevation"]
lats = ds_gebco["lat"]
lons = ds_gebco["lon"]

# ---------------------------------------------------------------------
# LOAD TPW EQUILIBRIUM ANOMALY (GeoJSON)
# ---------------------------------------------------------------------

with open(TPW_GEOJSON, "r") as f:
    tpw_data = json.load(f)

tpw_lons = np.array([f["geometry"]["coordinates"][0]
                      for f in tpw_data["features"]])
tpw_lats = np.array([f["geometry"]["coordinates"][1]
                      for f in tpw_data["features"]])
tpw_vals = np.array([f["properties"]["net_relative_sea_level_m"]
                      for f in tpw_data["features"]])

# ---------------------------------------------------------------------
# INTERPOLATE TPW ANOMALY ONTO GEBCO GRID
# ---------------------------------------------------------------------

grid_lon, grid_lat = np.meshgrid(lons.values, lats.values)

tpw_interp = griddata(
    points=(tpw_lons, tpw_lats),
    values=tpw_vals,
    xi=(grid_lon, grid_lat),
    method="linear"
)

# Conservative fill outside convex hull
tpw_interp = np.nan_to_num(tpw_interp, nan=0.0)

# ---------------------------------------------------------------------
# COMPUTE EFFECTIVE ELEVATION
# ---------------------------------------------------------------------

effective_elevation = elev.values - tpw_interp

# ---------------------------------------------------------------------
# BUILD OUTPUT DATASET
# ---------------------------------------------------------------------

ds_out = xr.Dataset(
    data_vars={
        "effective_elevation": (("lat", "lon"), effective_elevation)
    },
    coords={
        "lat": lats,
        "lon": lons
    },
    attrs={
        "title": "Realized Effective Elevation Under TPW Equilibrium Sea Level",
        "summary": (
            "Effective elevation computed as GEBCO sub-ice bedrock elevation "
            "minus modeled equilibrium relative sea-level anomaly following "
            "a 104-degree true polar wander scenario."
        ),
        "units": "meters",
        "convention": "Positive = emergent land; Negative = submerged",
        "gebco_source": "GEBCO 2025 sub-ice topography/bathymetry",
        "tpw_source": "Modeled equilibrium relative sea-level anomaly",
        "crs": "EPSG:4326",
        "history": "Generated by sealevel-gebco.py"
    }
)

ds_out["effective_elevation"].attrs = {
    "long_name": "Effective elevation relative to equilibrium sea level",
    "units": "m"
}

# ---------------------------------------------------------------------
# WRITE NETCDF (compressed)
# ---------------------------------------------------------------------

encoding = {
    "effective_elevation": {
        "zlib": True,
        "complevel": 4,
        "dtype": "float32"
    }
}

ds_out.to_netcdf(
    OUT_NC,
    format="NETCDF4",
    encoding=encoding
)

print("--------------------------------------------------")
print("GEBCO–TPW integration complete")
print(f"NetCDF output written to: {OUT_NC}")
print("Variable: effective_elevation (meters)")
print("  > 0  : emergent land")
print("  < 0  : submerged")
print("--------------------------------------------------")
