#!/user/bin/env python3

import json
import numpy as np
import matplotlib.pyplot as plt

import cartopy.crs as ccrs
import cartopy.feature as cfeature

from scipy.interpolate import griddata
from matplotlib.colors import BoundaryNorm

# ------------------------------------------------------------
# Load sea-level anomaly GeoJSON
# ------------------------------------------------------------

sl_file = "TPW_signed_equilibrium_relative_sea_level.geojson"

sl_lats = []
sl_lons = []
sl_vals = []

with open(sl_file, "r") as f:
    sl_data = json.load(f)

for feat in sl_data["features"]:
    lon, lat = feat["geometry"]["coordinates"]
    val = feat["properties"]["net_relative_sea_level_m"]

    sl_lons.append(lon)
    sl_lats.append(lat)
    sl_vals.append(val)

sl_lons = np.array(sl_lons)
sl_lats = np.array(sl_lats)
sl_vals = np.array(sl_vals)

# ------------------------------------------------------------
# Interpolate anomaly field to regular grid
# ------------------------------------------------------------

lon_grid = np.linspace(-180, 180, 361)
lat_grid = np.linspace(-90, 90, 181)

LON, LAT = np.meshgrid(lon_grid, lat_grid)

Z = griddata(
    (sl_lons, sl_lats),
    sl_vals,
    (LON, LAT),
    method="linear"
)

# ------------------------------------------------------------
# Discrete contour levels (meters)
# ------------------------------------------------------------

levels = np.array([
    -10000, -7500, -5000, -2500,
    -1000,  -500,  -250,
       0,
      250,   500,  1000,
     2500,  5000,  7500, 10000
])

norm = BoundaryNorm(levels, ncolors=256, clip=True)

# ------------------------------------------------------------
# Load Early Homo dataset
# ------------------------------------------------------------

homo_lons = []
homo_lats = []

with open("early_homo_sites.geojson", "r") as f:
    homo_data = json.load(f)

for feat in homo_data["features"]:
    lon, lat = feat["geometry"]["coordinates"]
    homo_lons.append(lon)
    homo_lats.append(lat)

homo_lons = np.array(homo_lons)
homo_lats = np.array(homo_lats)

# ------------------------------------------------------------
# Load Early Civilization dataset
# ------------------------------------------------------------

civ_lons = []
civ_lats = []

with open("early_civilizations.geojson", "r") as f:
    civ_data = json.load(f)

for feat in civ_data["features"]:
    lon, lat = feat["geometry"]["coordinates"]
    civ_lons.append(lon)
    civ_lats.append(lat)

civ_lons = np.array(civ_lons)
civ_lats = np.array(civ_lats)

# ------------------------------------------------------------
# Create map
# ------------------------------------------------------------

fig = plt.figure(figsize=(17, 9))
ax = plt.axes(projection=ccrs.Robinson())
ax.set_global()

ax.set_title(
    "Early Homo and Early Civilizations\n"
    "Projected onto Equilibrium Sea-Level Anomaly After 104° TPW",
    fontsize=14,
    pad=14
)

# ------------------------------------------------------------
# Plot sea-level anomaly contours
# ------------------------------------------------------------

cf = ax.contourf(
    LON,
    LAT,
    Z,
    levels=levels,
    cmap="RdBu_r",
    norm=norm,
    transform=ccrs.PlateCarree(),
    extend="both",
    alpha=0.9
)

ax.contour(
    LON,
    LAT,
    Z,
    levels=[0],
    colors="black",
    linewidths=1.0,
    transform=ccrs.PlateCarree()
)

# ------------------------------------------------------------
# Plot Early Homo sites
# ------------------------------------------------------------

ax.scatter(
    homo_lons,
    homo_lats,
    s=35,
    color="black",
    edgecolor="white",
    linewidth=0.5,
    transform=ccrs.PlateCarree(),
    label="Early Homo sites",
    zorder=5
)

# ------------------------------------------------------------
# Plot Early Civilization sites
# ------------------------------------------------------------

ax.scatter(
    civ_lons,
    civ_lats,
    s=40,
    color="gold",
    edgecolor="black",
    linewidth=0.5,
    transform=ccrs.PlateCarree(),
    label="Early civilizations",
    zorder=6
)

# ------------------------------------------------------------
# Geographic reference layers
# ------------------------------------------------------------

ax.add_feature(cfeature.COASTLINE, linewidth=0.6)
ax.add_feature(cfeature.BORDERS, linewidth=0.4)
ax.add_feature(cfeature.LAND, facecolor="none", edgecolor="black", linewidth=0.3)

# ------------------------------------------------------------
# Colorbar
# ------------------------------------------------------------

cbar = plt.colorbar(
    cf,
    orientation="horizontal",
    pad=0.06,
    fraction=0.06,
    ticks=levels
)

cbar.set_label("Net Relative Sea-Level Change (meters)", fontsize=11)

# ------------------------------------------------------------
# Legend
# ------------------------------------------------------------

ax.legend(
    loc="lower left",
    frameon=True,
    framealpha=0.9
)

# ------------------------------------------------------------
# Save and show
# ------------------------------------------------------------

plt.savefig(
    "TPW_sea_level_with_homo_and_civilizations.png",
    dpi=300,
    bbox_inches="tight"
)

plt.show()
