#!/usr/bin/env python3

import json
import numpy as np
import matplotlib.pyplot as plt

import cartopy.crs as ccrs
import cartopy.feature as cfeature

from scipy.interpolate import griddata
from matplotlib.colors import BoundaryNorm

# ---------------------------------------------------------------------
# Load GeoJSON anomaly data
# ---------------------------------------------------------------------

geojson_path = "TPW_signed_equilibrium_relative_sea_level.geojson"

lats = []
lons = []
anomaly = []

with open(geojson_path, "r") as f:
    data = json.load(f)

for feat in data["features"]:
    lon, lat = feat["geometry"]["coordinates"]
    val = feat["properties"]["net_relative_sea_level_m"]

    lons.append(lon)
    lats.append(lat)
    anomaly.append(val)

lons = np.array(lons)
lats = np.array(lats)
anomaly = np.array(anomaly)

# ---------------------------------------------------------------------
# Interpolate onto regular global grid
# ---------------------------------------------------------------------

lon_grid = np.linspace(-180, 180, 361)
lat_grid = np.linspace(-90, 90, 181)

LON, LAT = np.meshgrid(lon_grid, lat_grid)

Z = griddata(
    points=(lons, lats),
    values=anomaly,
    xi=(LON, LAT),
    method="linear"
)

# ---------------------------------------------------------------------
# Define discrete contour levels (meters)
# ---------------------------------------------------------------------

levels = np.array([
    -10000, -7500, -5000, -2500,
    -1000,  -500,  -250,
      0,
     250,   500,  1000,
    2500,  5000,  7500, 10000
])

norm = BoundaryNorm(levels, ncolors=256, clip=True)

# ---------------------------------------------------------------------
# Create map
# ---------------------------------------------------------------------

fig = plt.figure(figsize=(16, 8))
ax = plt.axes(projection=ccrs.Robinson())

ax.set_global()
ax.set_title(
    "Net Relative Sea-Level Anomaly After 104° True Polar Wander\n"
    "Discrete Equilibrium Contours (Positive = Submergence, Negative = Emergence)",
    fontsize=14,
    pad=12
)

# ---------------------------------------------------------------------
# Plot filled contours
# ---------------------------------------------------------------------

cf = ax.contourf(
    LON,
    LAT,
    Z,
    levels=levels,
    cmap="RdBu_r",
    norm=norm,
    transform=ccrs.PlateCarree(),
    extend="both"
)

# Optional: draw contour lines
cl = ax.contour(
    LON,
    LAT,
    Z,
    levels=levels,
    colors="black",
    linewidths=0.3,
    alpha=0.4,
    transform=ccrs.PlateCarree()
)

# ---------------------------------------------------------------------
# Geographic reference layers
# ---------------------------------------------------------------------

ax.add_feature(cfeature.COASTLINE, linewidth=0.6)
ax.add_feature(cfeature.BORDERS, linewidth=0.4)
ax.add_feature(cfeature.LAND, facecolor="none", edgecolor="black", linewidth=0.3)

# ---------------------------------------------------------------------
# Colorbar
# ---------------------------------------------------------------------

cbar = plt.colorbar(
    cf,
    orientation="horizontal",
    pad=0.06,
    fraction=0.06,
    ticks=levels
)

cbar.set_label("Net Relative Sea-Level Change (meters)", fontsize=11)

# ---------------------------------------------------------------------
# Save and show
# ---------------------------------------------------------------------

plt.savefig(
    "TPW_signed_relative_sea_level_contours.png",
    dpi=300,
    bbox_inches="tight"
)

plt.show()
