#!/usr/bin/env python3
import pandas as pd
import matplotlib.pyplot as plt
import numpy as np

# --------------------------------------------------
# LOAD RESULTS
# --------------------------------------------------

df = pd.read_csv("spatial.csv")

# Ensure correct ordering
df = df.sort_values("L_km")

scales = df["L_km"].values
I_obs = df["I_obs"].values
I_null = df["I_null_mean"].values

# --------------------------------------------------
# PLOT
# --------------------------------------------------

plt.figure(figsize=(7, 5))

# Observed Moran's I
plt.plot(
    scales,
    I_obs,
    marker="o",
    linewidth=2.2,
    markersize=7,
    color="black",
    label="Observed Moran’s I"
)

# Null expectation (≈ 0)
plt.plot(
    scales,
    I_null,
    linestyle="--",
    linewidth=1.5,
    color="gray",
    label="Permutation null mean"
)

# Zero reference line
plt.axhline(
    0,
    color="lightgray",
    linewidth=1,
    linestyle=":"
)

# --------------------------------------------------
# AXES FORMATTING
# --------------------------------------------------

plt.xscale("log")
plt.xticks(
    scales,
    [f"{int(s)}" for s in scales]
)

plt.xlabel("Spatial scale (km)")
plt.ylabel("Moran’s I")

plt.title("Scale Dependence of Spatial Autocorrelation")

plt.legend(frameon=False)

plt.tight_layout()
plt.savefig(
    "morans_I_vs_scale.png",
    dpi=300,
    bbox_inches="tight"
)
plt.show()
