#!/usr/bin/env python3

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

# ------------------------------------------------------------
# 1. LOAD RESULTS FILE
# ------------------------------------------------------------

df = pd.read_csv("Silver1996_shear_misfit_results.csv")

phi_obs = df["phi_deg"].values % 180.0
net1_at_sks = df["shear_net1_azimuth"].values
obs_misfit = df["angular_misfit_deg"].values


# ------------------------------------------------------------
# 2. AXIAL MISFIT FUNCTION
# ------------------------------------------------------------

def axial_misfit(theta_obs, theta_model):
    d = np.abs(theta_obs - theta_model) % 180.0
    return np.minimum(d, 180.0 - d)


# ------------------------------------------------------------
# 3. BUILD NULL CDF ENVELOPE (SINGLE-NET)
# ------------------------------------------------------------

K = 10000
x = np.linspace(0, 90, 500)

null_cdf = np.zeros((K, len(x)))

for k in range(K):
    alpha = np.random.uniform(0.0, 180.0)
    rotated_phi = (phi_obs + alpha) % 180.0
    misfit_k = axial_misfit(rotated_phi, net1_at_sks)

    misfit_k = np.sort(misfit_k)
    null_cdf[k] = np.searchsorted(misfit_k, x, side="right") / len(misfit_k)

null_lo = np.percentile(null_cdf, 5, axis=0)
null_hi = np.percentile(null_cdf, 95, axis=0)
null_med = np.percentile(null_cdf, 50, axis=0)


# ------------------------------------------------------------
# 4. OBSERVED CDF
# ------------------------------------------------------------

obs_sorted = np.sort(obs_misfit)
obs_cdf = np.searchsorted(obs_sorted, x, side="right") / len(obs_sorted)


# ------------------------------------------------------------
# 5. PLOT
# ------------------------------------------------------------

plt.figure(figsize=(7, 4.5))

plt.fill_between(
    x,
    null_lo,
    null_hi,
    color="lightgray",
    alpha=0.85,
    label="Null envelope (5–95%)"
)

plt.plot(
    x,
    null_med,
    color="gray",
    linestyle="--",
    linewidth=1.5,
    label="Null median"
)

plt.plot(
    x,
    obs_cdf,
    color="tab:orange",
    linewidth=2.5,
    label="Observed (Silver 1996)"
)

plt.axvline(
    obs_misfit.mean(),
    color="tab:orange",
    linestyle=":",
    linewidth=2,
    label=f"Observed mean = {obs_misfit.mean():.1f}°"
)

plt.axvline(
    45,
    color="black",
    linestyle=":",
    linewidth=1,
    label="Random axial mean (45°)"
)

plt.xlabel("Axial misfit angle (degrees)")
plt.ylabel("Cumulative probability")
plt.xlim(0, 90)
plt.ylim(0, 1)

plt.title("CDF of SKS Fast-Axis Misfit vs Shear-Net Geometry")
plt.legend(frameon=False)
plt.tight_layout()
plt.show()
