import pandas as pd

# === Configuration ===
INPUT_CSV = "silver1996.csv"
OUTPUT_KML = "silver1996.kml"
DELIMITER = ';'  # Semicolon-separated CSV

# Color thresholds for delay time (delta_t_s in seconds)
def get_style_name(delay):
    if pd.isna(delay) or delay == 0:
        return "zero"
    if delay <= 0.7:
        return "low"
    if delay <= 1.3:
        return "medium"
    return "high"

# Color scheme (KML ABGR format: aabbggrr)
STYLE_COLORS = {
    "low":    "ff00ff00",    # green
    "medium": "ff00ffff",    # yellow
    "high":   "ff0000ff",    # red
    "zero":   "ffaaaaaa"     # gray
}

ICON_URLS = {
    "low":    "http://maps.google.com/mapfiles/kml/paddle/grn-circle.png",
    "medium": "http://maps.google.com/mapfiles/kml/paddle/ylw-circle.png",
    "high":   "http://maps.google.com/mapfiles/kml/paddle/red-circle.png",
    "zero":   "http://maps.google.com/mapfiles/kml/paddle/wht-blank.png"
}

# === Read data ===
df = pd.read_csv(INPUT_CSV, sep=DELIMITER)

# Make sure we have the needed columns
required = ['Station', 'Latitude', 'Longitude', 'phi_deg', 'delta_t_s', 'Continent', 'Tectonic', 'Ref']
if not all(col in df.columns for col in required):
    print("Missing required columns!")
    exit(1)

# Convert to numeric where appropriate, force errors → NaN
for col in ['Latitude', 'Longitude', 'phi_deg', 'delta_t_s']:
    df[col] = pd.to_numeric(df[col], errors='coerce')

# === Start building KML ===
kml_lines = [
    '<?xml version="1.0" encoding="UTF-8"?>',
    '<kml xmlns="http://www.opengis.net/kml/2.2">',
    '<Document>',
    '  <name>Shear-wave splitting - Silver et al. 1996</name>',
    '  <description>Seismic stations with splitting parameters</description>',
    '  <open>1</open>',
    ''
]

# Define styles
for style_name in ["low", "medium", "high", "zero"]:
    kml_lines.extend([
        f'  <Style id="{style_name}">',
        '    <IconStyle>',
        f'      <color>{STYLE_COLORS[style_name]}</color>',
        '      <scale>1.1</scale>',
        '      <Icon>',
        f'        <href>{ICON_URLS[style_name]}</href>',
        '      </Icon>',
        '    </IconStyle>',
        '    <LabelStyle>',
        '      <scale>0.9</scale>',
        '    </LabelStyle>',
        '  </Style>',
        ''
    ])

kml_lines.append('  <Folder>')
kml_lines.append('    <name>Stations</name>')
kml_lines.append('    <open>1</open>')

# Create placemarks with ExtendedData for additional fields
for _, row in df.iterrows():
    station = str(row['Station']).strip()
    lat = float(row['Latitude'])
    lon = float(row['Longitude'])
    phi = row['phi_deg']
    delay = row['delta_t_s']
    continent = str(row['Continent']).strip()
    tectonic = str(row['Tectonic']).strip()
    ref = str(row['Ref']).strip()

    style = get_style_name(delay)

    # Build ExtendedData
    extended_data = [
        '      <ExtendedData>',
        f'        <Data name="phi_deg"><value>{phi}</value></Data>',
        f'        <Data name="delta_t_s"><value>{delay:.2f}</value></Data>',
        f'        <Data name="Continent"><value>{continent}</value></Data>',
        f'        <Data name="Tectonic"><value>{tectonic}</value></Data>',
        f'        <Data name="Ref"><value>{ref}</value></Data>',
        '      </ExtendedData>'
    ]

    placemark = [
        '    <Placemark>',
        f'      <name>{station}</name>',
        *extended_data,
        f'      <styleUrl>#{style}</styleUrl>',
        '      <Point>',
        f'        <coordinates>{lon},{lat},0</coordinates>',
        '      </Point>',
        '    </Placemark>'
    ]

    kml_lines.extend(placemark)

# Close tags
kml_lines.extend([
    '  </Folder>',
    '</Document>',
    '</kml>'
])

# === Write file ===
with open(OUTPUT_KML, 'w', encoding='utf-8') as f:
    f.write('\n'.join(kml_lines))

print(f"KML file successfully created: {OUTPUT_KML}")
print(f"Total stations written: {len(df)}")