#!/usr/bin/env python3

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

# ------------------------------------------------------------
# 1. LOAD RESULTS FILE
# ------------------------------------------------------------

df = pd.read_csv("Silver1996_shear_misfit_results.csv")

phi_obs = df["phi_deg"].values % 180.0
net1_at_sks = df["shear_net1_azimuth"].values
obs_misfit = df["angular_misfit_deg"].values


# ------------------------------------------------------------
# 2. AXIAL MISFIT FUNCTION
# ------------------------------------------------------------

def axial_misfit(theta_obs, theta_model):
    d = np.abs(theta_obs - theta_model) % 180.0
    return np.minimum(d, 180.0 - d)


# ------------------------------------------------------------
# 3. BUILD NULL HISTOGRAM ENVELOPE (SINGLE-NET)
# ------------------------------------------------------------

K = 10000
bins = np.linspace(0, 90, 31)
bin_centers = 0.5 * (bins[:-1] + bins[1:])

null_hist = np.zeros((K, len(bin_centers)))

for k in range(K):
    alpha = np.random.uniform(0.0, 180.0)
    rotated_phi = (phi_obs + alpha) % 180.0
    misfit_k = axial_misfit(rotated_phi, net1_at_sks)
    h, _ = np.histogram(misfit_k, bins=bins, density=True)
    null_hist[k] = h

null_lo = np.percentile(null_hist, 5, axis=0)
null_hi = np.percentile(null_hist, 95, axis=0)
null_med = np.percentile(null_hist, 50, axis=0)


# ------------------------------------------------------------
# 4. OBSERVED HISTOGRAM
# ------------------------------------------------------------

obs_hist, _ = np.histogram(obs_misfit, bins=bins, density=True)


# ------------------------------------------------------------
# 5. PLOT
# ------------------------------------------------------------

plt.figure(figsize=(7, 4.5))

plt.fill_between(
    bin_centers,
    null_lo,
    null_hi,
    color="lightgray",
    alpha=0.85,
    label="Null envelope (5–95%)"
)

plt.plot(
    bin_centers,
    null_med,
    color="gray",
    linestyle="--",
    linewidth=1.5,
    label="Null median"
)

plt.step(
    bin_centers,
    obs_hist,
    where="mid",
    color="tab:orange",
    linewidth=2.2,
    label="Observed (Silver 1996)"
)

plt.axvline(
    obs_misfit.mean(),
    color="tab:orange",
    linestyle=":",
    linewidth=2,
    label=f"Observed mean = {obs_misfit.mean():.1f}°"
)

plt.axvline(
    45,
    color="black",
    linestyle=":",
    linewidth=1,
    label="Random axial mean (45°)"
)

plt.xlabel("Axial misfit angle (degrees)")
plt.ylabel("Probability density")
plt.xlim(0, 90)
plt.title("SKS Fast-Axis Misfit vs Shear-Net Geometry")

plt.legend(frameon=False)
plt.tight_layout()
plt.show()
